(function($){
  function safeString(v){ return (v === null || v === undefined) ? '' : String(v); }

  function normalizeName(name){
    return (name || '')
      .toLowerCase()
      .replace(/[^a-z0-9_]+/g, '_')
      .replace(/^_+|_+$/g, '') || ('field_' + Math.random().toString(36).slice(2,8));
  }

  function parseSchema(raw){
    try{
      const s = JSON.parse(raw || '{}');
      if (!s || typeof s !== 'object') return {version:1, fields:[]};
      if (!Array.isArray(s.fields)) s.fields = [];
      if (!s.version) s.version = 1;
      if (!s.layout || typeof s.layout !== 'object') s.layout = {rows:[], unassigned:[]};
      if (!Array.isArray(s.layout.rows)) s.layout.rows = [];
      if (!Array.isArray(s.layout.unassigned)) s.layout.unassigned = [];
      return s;
    }catch(e){
      return {version:1, fields:[], layout:{rows:[], unassigned:[]}};
    }
  }

  function fieldSummary(f){
    const t = safeString(f.type || 'text');
    const label = safeString(f.label || '');
    const name = safeString(f.name || '');
    const req = f.required ? ' *' : '';
    return label + req + (name ? (' — ' + name) : '') + ' (' + t + ')';
  }

  function ensureLayout(schema){
    if (!schema.layout || typeof schema.layout !== 'object') schema.layout = {rows:[], unassigned:[]};
    if (!Array.isArray(schema.layout.rows)) schema.layout.rows = [];
    if (!Array.isArray(schema.layout.unassigned)) schema.layout.unassigned = [];
    reconcileLayout(schema);
  }

  function reconcileLayout(schema){
    const ids = schema.fields.map(f => safeString(f.id)).filter(Boolean);
    const valid = {};
    ids.forEach(id => { valid[id] = true; });

    const seen = {};
    const rows = [];

    (schema.layout.rows || []).forEach(row => {
      if (!row || typeof row !== 'object') return;
      let cols = parseInt(row.cols || 0, 10);
      if (!cols || cols < 1) cols = Array.isArray(row.columns) ? row.columns.length : 1;
      cols = Math.max(1, Math.min(3, cols));

      const colsIn = Array.isArray(row.columns) ? row.columns : (Array.isArray(row.cells) ? row.cells : []);
      const colsOut = [];
      for (let c=0; c<cols; c++){
        const colIn = Array.isArray(colsIn[c]) ? colsIn[c] : [];
        const colOut = [];
        colIn.forEach(fid => {
          fid = safeString(fid);
          if (!fid || !valid[fid] || seen[fid]) return;
          seen[fid] = true;
          colOut.push(fid);
        });
        colsOut.push(colOut);
      }

      rows.push({cols: cols, columns: colsOut});
    });

    const unassigned = [];
    (schema.layout.unassigned || []).forEach(fid => {
      fid = safeString(fid);
      if (!fid || !valid[fid] || seen[fid]) return;
      seen[fid] = true;
      unassigned.push(fid);
    });

    // Append any remaining fields not referenced.
    ids.forEach(fid => {
      if (!seen[fid]) unassigned.push(fid);
    });

    schema.layout.rows = rows;
    schema.layout.unassigned = unassigned;
  }

  $(function(){
    const $hidden = $('#warforms-schema-json');
    const $list = $('#wf_fields_list');
    const $listBuilder = $('#wf_fields_list_builder');
    const $canvas = $('#wf_layout_canvas');
    const $type = $('#wf_field_type');
    const $label = $('#wf_field_label');
    const $name = $('#wf_field_name');
    const $req = $('#wf_field_required');
    const $opts = $('#wf_field_options');
    const $curr = $('#wf_field_currencies');
    const $status = $('#wf_status');
    const $cancel = $('#wf_cancel_edit');
    const $add = $('#wf_add_field');

    const $schemaEditor = $('#wf_schema_editor');
    const $applyJson = $('#wf_apply_json');
    const $jsonStatus = $('#wf_json_status');

    let schema = parseSchema($hidden.val());
    let editingId = null;

    function syncHidden(){
      ensureLayout(schema);
      $hidden.val(JSON.stringify(schema));
      // keep the advanced editor up to date unless user is actively editing it
      if (!$schemaEditor.is(':focus')){
        $schemaEditor.val(JSON.stringify(schema, null, 2));
      }
    }

    function showConditional(){
      const t = safeString($type.val());
      $('.wf-for-options').toggle(['select','radio','multiselect','checkbox_group'].indexOf(t) !== -1);
      $('.wf-for-currency').toggle(t === 'currency');
    }

    function clearEditor(){
      editingId = null;
      $type.val('text');
      $label.val('');
      $name.val('');
      $req.prop('checked', false);
      $opts.val('');
      $curr.val('');
      $cancel.hide();
      $add.text('Add Field');
      showConditional();
    }

    function startEdit(fieldId){
      const f = schema.fields.find(x => x && x.id === fieldId);
      if (!f) return;
      editingId = fieldId;
      $type.val(f.type || 'text');
      $label.val(f.label || '');
      $name.val(f.name || '');
      $req.prop('checked', !!f.required);
      if (Array.isArray(f.options)) {
        $opts.val(f.options.join("\n"));
      } else if (typeof f.options === 'string') {
        $opts.val(f.options);
      } else {
        $opts.val('');
      }
      if (Array.isArray(f.allowed)) {
        $curr.val(f.allowed.join(', '));
      } else {
        $curr.val('');
      }
      $cancel.show();
      $add.text('Save Field');
      showConditional();
      $status.text('Editing…');
      setTimeout(() => $status.text(''), 1200);
    }

    function delField(fieldId){
      if (!confirm('Delete this field?')) return;
      schema.fields = schema.fields.filter(f => f && f.id !== fieldId);
      // remove from layout/unassigned
      ensureLayout(schema);
      schema.layout.rows.forEach(r => {
        r.columns = r.columns.map(col => col.filter(id => id !== fieldId));
      });
      schema.layout.unassigned = schema.layout.unassigned.filter(id => id !== fieldId);
      if (editingId === fieldId) clearEditor();
      syncHidden();
      render();
    }

    function buildField(existingId){
      const t = safeString($type.val() || 'text');
      const label = safeString($label.val()).trim();
      const nameRaw = safeString($name.val()).trim();
      const name = normalizeName(nameRaw || label);
      const required = !!$req.is(':checked');

      if (!label){
        alert('Label is required.');
        return null;
      }

      const field = {
        id: existingId || ('f_' + Date.now() + '_' + Math.random().toString(36).slice(2,6)),
        type: t,
        label,
        name,
        required
      };

      if (['select','radio','multiselect','checkbox_group'].indexOf(t) !== -1){
        const lines = safeString($opts.val()).split(/\r?\n/).map(s => s.trim()).filter(Boolean);
        field.options = lines;
      }

      if (t === 'currency'){
        const allowed = safeString($curr.val()).split(',').map(s => s.trim().toUpperCase()).filter(Boolean);
        field.allowed = allowed.length ? allowed : ['USD'];
      }

      return field;
    }

    function addOrSave(){
      const field = buildField(editingId);
      if (!field) return;

      const idx = schema.fields.findIndex(f => f && f.id === field.id);
      if (idx >= 0){
        schema.fields[idx] = field;
      } else {
        schema.fields.push(field);
        ensureLayout(schema);
        schema.layout.unassigned.push(field.id);
      }

      // Ensure layout consistency and avoid dupes.
      ensureLayout(schema);
      syncHidden();

      $status.text(editingId ? 'Saved.' : 'Added.');
      setTimeout(() => $status.text(''), 1200);
      clearEditor();
      render();
    }

    function renderItem(fid){
      const f = schema.fields.find(x => x && x.id === fid);
      if (!f) return '';
      const label = safeString(f.label || '(no label)');
      const type = safeString(f.type || 'text');
      const req = f.required ? ' *' : '';
      return (
        '<div class="wf-item" data-field-id="' + escapeHtml(fid) + '">' +
          '<div class="wf-item-meta">' + escapeHtml(label + req) +
            '<span class="wf-item-sub">(' + escapeHtml(type) + ')</span>' +
          '</div>' +
          '<div class="wf-item-actions">' +
            '<button type="button" class="button button-small wf-item-edit">Edit</button>' +
            '<button type="button" class="button button-small wf-item-del">Delete</button>' +
          '</div>' +
        '</div>'
      );
    }

    function escapeHtml(s){
      return safeString(s)
        .replace(/&/g,'&amp;')
        .replace(/</g,'&lt;')
        .replace(/>/g,'&gt;')
        .replace(/"/g,'&quot;')
        .replace(/'/g,'&#039;');
    }

    function renderLayout(){
      ensureLayout(schema);

      $canvas.empty();

      if (!schema.layout.rows.length){
        $canvas.append('<p class="description" style="margin-top:0">No rows yet. Add a row and drag fields into columns.</p>');
      }

      schema.layout.rows.forEach((row, rIdx) => {
        const cols = Math.max(1, Math.min(3, parseInt(row.cols || 1, 10)));
        const $row = $('<div class="wf-layout-row" data-row="'+rIdx+'">');

        const $head = $('<div class="wf-layout-row-head">');
        $head.append('<strong>Row ' + (rIdx+1) + '</strong>');

        const $controls = $('<div class="wf-layout-row-controls">');

        const $sel = $('<select class="wf-row-cols"></select>')
          .append('<option value="1">1 col</option>')
          .append('<option value="2">2 cols</option>')
          .append('<option value="3">3 cols</option>')
          .val(String(cols))
          .on('change', function(){
            setRowCols(rIdx, parseInt($(this).val(), 10));
          });

        const $remove = $('<button type="button" class="button">Remove</button>')
          .on('click', function(){ removeRow(rIdx); });

        $controls.append($sel).append(' ').append($remove);
        $head.append($controls);
        $row.append($head);

        const $cols = $('<div class="wf-layout-cols wf-cols-'+cols+'"></div>');
        for (let c=0; c<cols; c++){
          const $dz = $('<div class="wf-dropzone wf-sortable" data-row="'+rIdx+'" data-col="'+c+'" data-placeholder="Drop fields here"></div>');
          const ids = Array.isArray(row.columns && row.columns[c]) ? row.columns[c] : [];
          ids.forEach(fid => {
            $dz.append(renderItem(fid));
          });
          if (!ids.length) $dz.addClass('wf-empty');
          $cols.append($dz);
        }
        $row.append($cols);
        $canvas.append($row);
      });

      // Unassigned fields list
      const $un = $('<div class="wf-dropzone wf-sortable" id="wf_unassigned" data-placeholder="Drag fields into the layout above"></div>');
      (schema.layout.unassigned || []).forEach(fid => {
        $un.append(renderItem(fid));
      });
      if (!(schema.layout.unassigned||[]).length) $un.addClass('wf-empty');

      $list.empty();
      $list.append('<p class="description" style="margin-top:0">Drag to assign fields into the layout. Unassigned fields render after the layout in a single column.</p>');
      $list.append($un);

      // Also render a non-draggable field list in the Form Builder tab (if present).
      if ($listBuilder.length){
        $listBuilder.empty();
        $listBuilder.append('<p class="description" style="margin-top:0">Fields in this form:</p>');
        const $simple = $('<div class="wf-fields-simple"></div>');
        (schema.fields || []).forEach(f => {
          if (!f || !f.id) return;
          $simple.append(renderItem(String(f.id)));
        });
        if (!(schema.fields||[]).length) $simple.addClass('wf-empty');
        $listBuilder.append($simple);
      }

      initSortables();
      bindItemActions();
    }

    function bindItemActions(){
      $('.wf-item-edit').off('click').on('click', function(){
        const fid = $(this).closest('.wf-item').data('field-id');
        startEdit(String(fid));
      });
      $('.wf-item-del').off('click').on('click', function(){
        const fid = $(this).closest('.wf-item').data('field-id');
        delField(String(fid));
      });
    }

    function initSortables(){
      const $sortables = $('.wf-sortable');
      $sortables.each(function(){
        const $el = $(this);
        if ($el.data('ui-sortable')) { $el.sortable('destroy'); }
      });

      $sortables.sortable({
        connectWith: '.wf-sortable',
        items: '> .wf-item',
        placeholder: 'wf-sort-highlight',
        forcePlaceholderSize: true,
        tolerance: 'pointer',
        start: function(e, ui){
          ui.placeholder.height(ui.item.outerHeight());
        },
        update: function(){
          syncFromDOM();
        },
        receive: function(){
          syncFromDOM();
        },
        remove: function(){
          syncFromDOM();
        }
      });
    }

    function syncFromDOM(){
      // rebuild layout from DOM order
      const rows = [];
      $('#wf_layout_canvas .wf-layout-row').each(function(){
        const $row = $(this);
        const cols = $row.find('.wf-layout-cols').hasClass('wf-cols-3') ? 3 :
                     $row.find('.wf-layout-cols').hasClass('wf-cols-2') ? 2 : 1;
        const columns = [];
        for (let c=0; c<cols; c++){
          const ids = [];
          $row.find('.wf-dropzone[data-col="'+c+'"] .wf-item').each(function(){
            ids.push(String($(this).data('field-id')));
          });
          columns.push(ids);
        }
        rows.push({cols: cols, columns: columns});
      });

      const unassigned = [];
      $('#wf_unassigned .wf-item').each(function(){
        unassigned.push(String($(this).data('field-id')));
      });

      schema.layout.rows = rows;
      schema.layout.unassigned = unassigned;
      reconcileLayout(schema);
      syncHidden();
      renderLayout(); // re-render to update empty placeholders and indexes
    }

    function setRowCols(rowIdx, newCols){
      ensureLayout(schema);
      newCols = Math.max(1, Math.min(3, parseInt(newCols || 1, 10)));

      const row = schema.layout.rows[rowIdx];
      if (!row) return;

      const oldCols = Math.max(1, Math.min(3, parseInt(row.cols || (row.columns ? row.columns.length : 1), 10)));
      let columns = Array.isArray(row.columns) ? row.columns : [];

      // Normalize columns length to oldCols.
      while (columns.length < oldCols) columns.push([]);
      if (columns.length > oldCols) columns = columns.slice(0, oldCols);

      if (newCols < oldCols){
        // move fields from removed columns into unassigned
        for (let c=newCols; c<oldCols; c++){
          schema.layout.unassigned = schema.layout.unassigned.concat(columns[c] || []);
        }
        columns = columns.slice(0, newCols);
      } else if (newCols > oldCols){
        for (let c=oldCols; c<newCols; c++){
          columns.push([]);
        }
      }

      row.cols = newCols;
      row.columns = columns;

      reconcileLayout(schema);
      syncHidden();
      render();
    }

    function removeRow(rowIdx){
      ensureLayout(schema);
      const row = schema.layout.rows[rowIdx];
      if (!row) return;
      // move all row fields to unassigned
      (row.columns || []).forEach(col => {
        if (Array.isArray(col)) schema.layout.unassigned = schema.layout.unassigned.concat(col);
      });
      schema.layout.rows.splice(rowIdx, 1);
      reconcileLayout(schema);
      syncHidden();
      render();
    }

    function addRow(cols){
      ensureLayout(schema);
      cols = Math.max(1, Math.min(3, parseInt(cols || 1, 10)));
      const columns = [];
      for (let c=0; c<cols; c++) columns.push([]);
      schema.layout.rows.push({cols: cols, columns: columns});
      syncHidden();
      render();
    }

    function clearLayout(){
      if (!confirm('Clear layout rows? Fields will remain and be unassigned.')) return;
      ensureLayout(schema);
      schema.layout.rows = [];
      schema.layout.unassigned = schema.fields.map(f => safeString(f.id)).filter(Boolean);
      reconcileLayout(schema);
      syncHidden();
      render();
    }

    function autoFill(){
      ensureLayout(schema);
      if (!schema.layout.rows.length){
        addRow(1);
        ensureLayout(schema);
      }
      if (!schema.layout.unassigned.length) return;

      let un = schema.layout.unassigned.slice();
      schema.layout.unassigned = [];

      schema.layout.rows.forEach(row => {
        const cols = row.cols || (row.columns ? row.columns.length : 1);
        while (row.columns.length < cols) row.columns.push([]);
        while (un.length){
          // Place next unassigned into the column with the fewest items.
          let best = 0;
          for (let c=1; c<cols; c++){
            if ((row.columns[c]||[]).length < (row.columns[best]||[]).length) best = c;
          }
          row.columns[best].push(un.shift());
          // keep filling until next row? we stop when columns are roughly balanced, but keep going
          if (!un.length) break;
          // small balancing: if row becomes too long, move to next row
          const total = row.columns.reduce((a, col) => a + (col ? col.length : 0), 0);
          if (total >= cols * 4) break;
        }
      });

      // Any remaining stay unassigned
      schema.layout.unassigned = schema.layout.unassigned.concat(un);

      reconcileLayout(schema);
      syncHidden();
      render();
    }

    function render(){
      showConditional();
      renderLayout();
      syncHidden();
    }

    // Wire controls
    $type.on('change', showConditional);
    $add.on('click', addOrSave);
    $cancel.on('click', function(){ clearEditor(); });

    $(document).on('click', '.wf-add-row', function(){
      addRow($(this).data('cols'));
    });
    $(document).on('click', '.wf-layout-clear', function(){ clearLayout(); });
    $(document).on('click', '.wf-layout-autofill', function(){ autoFill(); });

    // Advanced JSON apply
    $applyJson.on('click', function(){
      try{
        schema = parseSchema($schemaEditor.val());
        ensureLayout(schema);
        syncHidden();
        render();
        $jsonStatus.text('Applied.');
        setTimeout(()=> $jsonStatus.text(''), 1500);
      }catch(e){
        $jsonStatus.text('Invalid JSON. Not applied.');
        setTimeout(()=> $jsonStatus.text(''), 2000);
      }
    });


    // Tabs: switch between Builder / Layout / Add-on panels.
    function initTabs(){
      const $links = $('.wf-tab-wrapper a[data-wf-tab]');
      const $panels = $('.wf-tab-panel');
      if (!$links.length || !$panels.length) return;

      const storageKey = 'warforms_admin_builder_tab';

      function activate(tab){
        tab = String(tab || '');
        if (!tab) tab = $links.first().data('wf-tab');

        // If the requested tab doesn't exist, fall back to first.
        if (!$links.filter('[data-wf-tab="'+tab+'"]').length){
          tab = $links.first().data('wf-tab');
        }

        $links.removeClass('nav-tab-active');
        $links.filter('[data-wf-tab="'+tab+'"]').addClass('nav-tab-active');

        $panels.removeClass('is-active').hide();
        $('#wf-tab-' + tab).addClass('is-active').show();

        try { if (window.localStorage) localStorage.setItem(storageKey, tab); } catch(e){}
      }

      $links.off('click.wftab').on('click.wftab', function(e){
        e.preventDefault();
        activate($(this).data('wf-tab'));
      });

      let initial = null;
      try { if (window.localStorage) initial = localStorage.getItem(storageKey); } catch(e){ initial = null; }
      activate(initial);
    }
    initTabs();

    // Initial paint
    ensureLayout(schema);
    render();
  });
})(jQuery);
