<?php
if (!defined('ABSPATH')) exit;

/**
 * WarForms builder metabox (tabbed).
 *
 * Tabs:
 * - Form Builder (add/edit fields + field list)
 * - Layout (rows/columns + drag/drop + advanced schema JSON)
 *
 * Add-ons can register additional tabs via:
 *   add_filter('warforms_admin_builder_tabs', function($tabs, $post_id){ ...; return $tabs; }, 10, 2);
 * and render panel via:
 *   add_action('warforms_admin_builder_render_tab_{tab_id}', function($post_id){ ... }, 10, 1);
 */

function warforms_reboot_admin_builder_supports_tabs(): bool {
    return true;
}

add_action('add_meta_boxes', function() {
    add_meta_box(
        'warforms_reboot_builder',
        'WarForms',
        'warforms_reboot_render_builder_metabox',
        'warform',
        'normal',
        'high'
    );
});

function warforms_reboot_render_builder_metabox(\WP_Post $post): void {
    if (!current_user_can('edit_post', $post->ID)) return;

    $schema = warforms_reboot_get_schema((int)$post->ID);
    $json   = wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    if (!is_string($json)) $json = '{"version":1,"fields":[]}';

    wp_nonce_field('warforms_reboot_save_schema', 'warforms_reboot_schema_nonce');

    // Tabs
    $tabs = [
        'builder' => ['label' => 'Form Builder', 'priority' => 10],
        'layout'  => ['label' => 'Layout',       'priority' => 20],
    ];
    $tabs = apply_filters('warforms_admin_builder_tabs', $tabs, (int)$post->ID);
    if (!is_array($tabs)) $tabs = [];

    // Ensure required core tabs exist
    if (!isset($tabs['builder'])) $tabs['builder'] = ['label' => 'Form Builder', 'priority' => 10];
    if (!isset($tabs['layout']))  $tabs['layout']  = ['label' => 'Layout',       'priority' => 20];

    // Normalize + sort by priority
    foreach ($tabs as $id => &$t) {
        if (!is_array($t)) $t = [];
        $t['label'] = isset($t['label']) ? (string)$t['label'] : (string)$id;
        $t['priority'] = isset($t['priority']) ? (int)$t['priority'] : 100;
    }
    unset($t);

    uasort($tabs, function($a, $b){
        $pa = (int)($a['priority'] ?? 100);
        $pb = (int)($b['priority'] ?? 100);
        if ($pa === $pb) return 0;
        return ($pa < $pb) ? -1 : 1;
    });

    echo '<div id="warforms-builder" class="warforms-reboot">';
    echo '<p style="margin-top:0">Build your form fields here. This plugin uses the Classic Editor for WarForms to ensure schema saves reliably.</p>';

    echo '<input type="hidden" id="warforms-schema-json" name="warforms_schema_json" value="' . esc_attr($json) . '">';

    // Tab nav
    echo '<h2 class="nav-tab-wrapper wf-tab-wrapper" style="margin-top:0">';
    $first = true;
    foreach ($tabs as $id => $t) {
        $active = $first ? ' nav-tab-active' : '';
        $first = false;
        echo '<a href="#" class="nav-tab' . $active . '" data-wf-tab="' . esc_attr($id) . '">' . esc_html($t['label']) . '</a>';
    }
    echo '</h2>';

    // Panels
    $firstPanel = true;
    foreach ($tabs as $id => $t) {
        $panelClasses = 'wf-tab-panel' . ($firstPanel ? ' is-active' : '');
        $style = $firstPanel ? '' : ' style="display:none"';
        $firstPanel = false;

        echo '<div class="' . esc_attr($panelClasses) . '" id="wf-tab-' . esc_attr($id) . '"' . $style . '>';

        if ($id === 'builder') {
            warforms_reboot_render_builder_tab_builder((int)$post->ID);
        } elseif ($id === 'layout') {
            warforms_reboot_render_builder_tab_layout((int)$post->ID, $json);
        } else {
            /**
             * Add-on tab render hook.
             * Usage: add_action('warforms_admin_builder_render_tab_{id}', ...)
             */
            do_action('warforms_admin_builder_render_tab_' . sanitize_key((string)$id), (int)$post->ID);
        }

        echo '</div>';
    }

    echo '</div>'; // .warforms-reboot
}

function warforms_reboot_render_builder_tab_builder(int $post_id): void {
    echo '<div class="wf-builder-grid">';

    // Add/Edit field panel
    echo '  <div class="wf-panel">';
    echo '    <h3 style="margin-top:0">Add / Edit Field</h3>';

    echo '    <label>Type<br><select id="wf_field_type">';
    $types = [
        'text'        => 'Text',
        'email'       => 'Email',
        'url'         => 'URL',
        'phone'       => 'Phone',
        'textarea'    => 'Textarea',
        'select'      => 'Select',
        'multiselect' => 'Multi-select',
        'radio'       => 'Radio',
        'checkbox_group' => 'Checkbox',
        'currency'    => 'Currency',
    ];
    foreach ($types as $k => $v) {
        echo '<option value="' . esc_attr($k) . '">' . esc_html($v) . '</option>';
    }
    echo '    </select></label><br><br>';

    echo '    <label>Label<br><input type="text" id="wf_field_label" style="width:100%" placeholder="e.g. Phone Number"></label><br><br>';
    echo '    <label>Name (slug)<br><input type="text" id="wf_field_name" style="width:100%" placeholder="e.g. phone"></label><br><br>';
    echo '    <label><input type="checkbox" id="wf_field_required"> Required</label><br><br>';

    echo '    <div class="wf-for-options" style="display:none">';
    echo '      <label>Options (one per line)<br><textarea id="wf_field_options" rows="5" style="width:100%"></textarea></label>';
    echo '    </div>';

    echo '    <div class="wf-for-currency" style="display:none">';
    echo '      <label>Currencies (one per line)<br><textarea id="wf_field_currencies" rows="5" style="width:100%"></textarea></label>';
    echo '      <p class="description">Example: USD, EUR, GBP. First currency is default.</p>';
    echo '    </div>';

    echo '    <p style="margin-top:12px">';
    echo '      <button type="button" class="button button-primary" id="wf_add_field">Add Field</button> ';
    echo '      <button type="button" class="button" id="wf_cancel_edit" style="display:none">Cancel</button>';
    echo '      <span class="wf-status" id="wf_status" style="margin-left:8px"></span>';
    echo '    </p>';

    echo '  </div>';

    // Fields list panel (non-draggable)
    echo '  <div class="wf-panel">';
    echo '    <h3 style="margin-top:0">Fields</h3>';
    echo '    <p class="description" style="margin-top:0">Edit or remove fields. Layout is configured in the Layout tab.</p>';
    echo '    <div id="wf_fields_list_builder"></div>';
    echo '  </div>';

    echo '</div>';
}

function warforms_reboot_render_builder_tab_layout(int $post_id, string $json): void {
    echo '<div class="wf-panel">';

    echo '  <h3 style="margin-top:0">Layout</h3>';
    echo '  <p class="description">Add rows and drag fields into columns. Fields not placed will render after the layout.</p>';

    echo '  <div class="wf-layout-toolbar">';
    echo '    <button type="button" class="button wf-add-row" data-cols="1">Add 1-column row</button> ';
    echo '    <button type="button" class="button wf-add-row" data-cols="2">Add 2-column row</button> ';
    echo '    <button type="button" class="button wf-add-row" data-cols="3">Add 3-column row</button> ';
    echo '    <button type="button" class="button wf-layout-autofill">Auto-fill</button> ';
    echo '    <button type="button" class="button wf-layout-clear">Clear</button>';
    echo '  </div>';

    echo '  <div id="wf_layout_canvas" class="wf-layout-canvas"></div>';

    echo '  <h3 style="margin-top:16px">Unassigned fields</h3>';
    echo '  <div id="wf_fields_list"></div>';

    echo '  <details style="margin-top:12px">';
    echo '    <summary><strong>Advanced: Schema JSON</strong></summary>';
    echo '    <p class="description">You can edit the schema JSON directly if you want. Click Apply to load it into the builder.</p>';
    echo '    <textarea id="wf_schema_editor" rows="10" style="width:100%">' . esc_textarea($json) . '</textarea>';
    echo '    <p><button type="button" class="button" id="wf_apply_json">Apply</button> <span class="wf-status" id="wf_json_status"></span></p>';
    echo '  </details>';

    echo '</div>';
}

add_action('save_post_warform', function($post_id, $post, $update) {
    if (!isset($_POST['warforms_reboot_schema_nonce']) || !wp_verify_nonce($_POST['warforms_reboot_schema_nonce'], 'warforms_reboot_save_schema')) {
        return;
    }
    if (!current_user_can('edit_post', $post_id)) return;

    $raw = isset($_POST['warforms_schema_json']) ? wp_unslash($_POST['warforms_schema_json']) : '';
    if (!is_string($raw) || trim($raw) === '') {
        warforms_reboot_set_schema((int)$post_id, ['version' => 1, 'fields' => []]);
        return;
    }

    $decoded = json_decode($raw, true);
    if (!is_array($decoded)) {
        // If bad JSON, keep existing schema and show an admin notice.
        set_transient('warforms_reboot_admin_notice_' . (int)$post_id, [
            'type' => 'error',
            'msg' => 'Schema JSON was invalid and was not saved.',
        ], 30);
        return;
    }

    // Minimal normalization/sanitization.
    $schema = [
        'version' => isset($decoded['version']) ? (int)$decoded['version'] : 1,
        'fields' => [],
    ];

    // Step 9: confirmation config in schema JSON
    if (isset($decoded['confirmation']) && is_array($decoded['confirmation'])) {
        $c = $decoded['confirmation'];
        $type = isset($c['type']) ? sanitize_key((string)$c['type']) : '';
        if (in_array($type, ['message','redirect'], true)) {
            $conf = ['type' => $type];

            if ($type === 'message') {
                $msg = isset($c['message']) ? sanitize_text_field((string)$c['message']) : '';
                if ($msg !== '') $conf['message'] = $msg;
            } elseif ($type === 'redirect') {
                $url = isset($c['redirect_url']) ? esc_url_raw((string)$c['redirect_url']) : '';
                if ($url !== '') $conf['redirect_url'] = $url;
            }

            $schema['confirmation'] = $conf;
        }
    }

    if (isset($decoded['fields']) && is_array($decoded['fields'])) {
        foreach ($decoded['fields'] as $f) {
            if (!is_array($f)) continue;
            $type = isset($f['type']) ? sanitize_key((string)$f['type']) : 'text';
            $name = isset($f['name']) ? warforms_reboot_normalize_field_name((string)$f['name']) : '';
            $label = isset($f['label']) ? sanitize_text_field((string)$f['label']) : '';
            if ($name === '' || $label === '') continue;

            $field = [
                'id' => isset($f['id']) ? sanitize_text_field((string)$f['id']) : ('f_' . wp_generate_password(8, false, false)),
                'type' => $type,
                'name' => $name,
                'label' => $label,
                'required' => !empty($f['required']),
            ];

            if (in_array($type, ['select','radio','multiselect','checkbox_group'], true)) {
                $opts = [];
                if (isset($f['options']) && is_array($f['options'])) {
                    foreach ($f['options'] as $o) {
                        $o = sanitize_text_field((string)$o);
                        if ($o !== '') $opts[] = $o;
                    }
                }
                $field['options'] = array_values(array_unique($opts));
            }

            if ($type === 'currency') {
                $allowed = [];
                if (isset($f['allowed']) && is_array($f['allowed'])) {
                    foreach ($f['allowed'] as $c) {
                        $c = strtoupper(preg_replace('/[^A-Z]/', '', (string)$c));
                        if (strlen($c) === 3) $allowed[] = $c;
                    }
                }
                if (empty($allowed)) $allowed = ['USD'];
                $field['allowed'] = array_values(array_unique($allowed));
            }

            $schema['fields'][] = $field;
        }
    }

    
    // Layout: rows/columns placement. Optional; if omitted, frontend falls back to field order.
    if (isset($decoded['layout']) && is_array($decoded['layout'])) {
        $layout_in = $decoded['layout'];

        // Build a map of valid field IDs.
        $valid_ids = [];
        foreach ($schema['fields'] as $_f) {
            if (is_array($_f) && !empty($_f['id'])) {
                $valid_ids[(string)$_f['id']] = true;
            }
        }

        $seen = [];
        $rows_out = [];

        $rows_in = [];
        if (isset($layout_in['rows']) && is_array($layout_in['rows'])) {
            $rows_in = $layout_in['rows'];
        } elseif (isset($layout_in[0]) && is_array($layout_in[0])) {
            // allow legacy layout as a numeric array of rows
            $rows_in = $layout_in;
        }

        foreach ($rows_in as $row) {
            if (!is_array($row)) continue;
            $cols = isset($row['cols']) ? (int)$row['cols'] : 0;
            $cols = max(1, min(3, $cols));

            $cols_in = [];
            if (isset($row['columns']) && is_array($row['columns'])) {
                $cols_in = $row['columns'];
            } elseif (isset($row['cells']) && is_array($row['cells'])) {
                $cols_in = $row['cells'];
            }

            // Normalize to exact number of columns.
            $columns_out = [];
            for ($i=0; $i<$cols; $i++) {
                $columns_out[$i] = [];
                $col_in = $cols_in[$i] ?? [];
                if (!is_array($col_in)) $col_in = [];
                foreach ($col_in as $fid) {
                    $fid = sanitize_text_field((string)$fid);
                    if ($fid === '' || empty($valid_ids[$fid]) || isset($seen[$fid])) continue;
                    $seen[$fid] = true;
                    $columns_out[$i][] = $fid;
                }
            }

            $rows_out[] = [
                'cols' => $cols,
                'columns' => $columns_out,
            ];
        }

        $unassigned_out = [];
        $un_in = isset($layout_in['unassigned']) && is_array($layout_in['unassigned']) ? $layout_in['unassigned'] : [];
        foreach ($un_in as $fid) {
            $fid = sanitize_text_field((string)$fid);
            if ($fid === '' || empty($valid_ids[$fid]) || isset($seen[$fid])) continue;
            $seen[$fid] = true;
            $unassigned_out[] = $fid;
        }

        // Append any remaining fields not referenced anywhere.
        foreach (array_keys($valid_ids) as $fid) {
            if (!isset($seen[$fid])) $unassigned_out[] = $fid;
        }

        $schema['layout'] = [
            'rows' => $rows_out,
            'unassigned' => $unassigned_out,
        ];
    }

update_post_meta($post_id, '_warforms_schema_json', wp_json_encode($schema, JSON_UNESCAPED_SLASHES));
}, 10, 3);

// Admin notices for schema save errors
add_action('admin_notices', function() {
    global $pagenow;
    if (!in_array($pagenow, ['post.php','post-new.php'], true)) return;
    $screen = get_current_screen();
    if (!$screen || $screen->post_type !== 'warform') return;
    $post_id = isset($_GET['post']) ? (int)$_GET['post'] : 0;
    if ($post_id <= 0) return;
    $notice = get_transient('warforms_reboot_admin_notice_' . $post_id);
    if (!$notice || !is_array($notice)) return;
    delete_transient('warforms_reboot_admin_notice_' . $post_id);

    $type = in_array($notice['type'] ?? '', ['error','warning','success','info'], true) ? $notice['type'] : 'info';
    $msg = $notice['msg'] ?? '';
    echo '<div class="notice notice-' . esc_attr($type) . ' is-dismissible"><p>' . esc_html($msg) . '</p></div>';
});

// Shortcode display + copy button in the Publish box.
add_action('post_submitbox_misc_actions', function() {
    global $post;
    if (!$post || $post->post_type !== 'warform') return;

    $shortcode = '[warform id="' . (int)$post->ID . '"]';

    echo '<div class="misc-pub-section warforms-reboot-shortcode">';
    echo '<label style="display:block; font-weight:600; margin-bottom:6px;">Shortcode</label>';
    echo '<div class="wf-shortcode-row">';
    echo '<input type="text" class="widefat wf-shortcode-input" readonly value="' . esc_attr($shortcode) . '">';
    echo '<button type="button" class="button wf-copy-shortcode" data-copy="' . esc_attr($shortcode) . '">Copy</button>';
    echo '</div>';
    echo '<div class="wf-shortcode-feedback" style="margin-top:6px;"></div>';
    echo '</div>';
});

// Enqueue admin assets only on warform edit screens.
add_action('admin_enqueue_scripts', function($hook) {
    $screen = get_current_screen();
    if (!$screen || $screen->post_type !== 'warform') return;

    wp_enqueue_script('jquery-ui-sortable');

    wp_enqueue_style('warforms-reboot-admin', WARFORMS_REBOOT_URL . 'assets/admin.css', [], WARFORMS_REBOOT_VERSION);
    wp_enqueue_script('warforms-reboot-admin', WARFORMS_REBOOT_URL . 'assets/admin.js', ['jquery'], WARFORMS_REBOOT_VERSION, true);
});
