<?php
if (!defined('ABSPATH')) exit;

function warforms_reboot_get_schema(int $form_id): array {
    $raw = get_post_meta($form_id, '_warforms_schema_json', true);
    if (!is_string($raw) || trim($raw) === '') {
        return ['version' => 1, 'fields' => []];
    }
    $decoded = json_decode($raw, true);
    if (!is_array($decoded)) {
        return ['version' => 1, 'fields' => []];
    }
    if (!isset($decoded['fields']) || !is_array($decoded['fields'])) {
        $decoded['fields'] = [];
    }
    if (!isset($decoded['version'])) {
        $decoded['version'] = 1;
    }
    return $decoded;
}

function warforms_reboot_set_schema(int $form_id, array $schema): void {
    update_post_meta($form_id, '_warforms_schema_json', wp_json_encode($schema, JSON_UNESCAPED_SLASHES));
}

function warforms_reboot_normalize_field_name(string $name): string {
    $name = strtolower($name);
    $name = preg_replace('/[^a-z0-9_]+/', '_', $name);
    $name = trim($name, '_');
    if ($name === '') $name = 'field_' . wp_generate_password(6, false, false);
    return $name;
}

function warforms_reboot_safe_redirect_back(array $args = []): void {
    $ref = wp_get_referer();
    if (!$ref) {
        $ref = home_url('/');
    }
    $url = add_query_arg($args, $ref);
    wp_safe_redirect($url);
    exit;
}

function warforms_reboot_set_last_error(string $code, string $detail = ''): void {
    update_option('warforms_reboot_last_error', [
        'time' => time(),
        'code' => sanitize_key($code),
        'detail' => sanitize_text_field($detail),
    ], false);
}

function warforms_reboot_get_last_error(): array {
    $v = get_option('warforms_reboot_last_error', []);
    return is_array($v) ? $v : [];
}
/**
 * Format a US/Canada (NANP) phone number for human-friendly display in admin.
 * Accepts values like +15551234567, 15551234567, 5551234567, (555)123-4567.
 * Returns AAA-BBB-CCCC when possible; otherwise returns the original trimmed value.
 */
function warforms_reboot_format_phone_display($value): string {
    if (is_array($value)) {
        // Best effort: stringify arrays.
        $value = wp_json_encode($value);
    }
    $s = trim((string) $value);
    if ($s === '') return '';

    $digits = preg_replace('/\D+/', '', $s);

    // E.164 +1XXXXXXXXXX -> digits length 11 starting with 1
    if (strlen($digits) === 11 && $digits[0] === '1') {
        $digits = substr($digits, 1);
    }

    // If it's 10 digits, format as AAA-BBB-CCCC.
    if (strlen($digits) === 10) {
        $a = substr($digits, 0, 3);
        $b = substr($digits, 3, 3);
        $c = substr($digits, 6, 4);
        return $a . '-' . $b . '-' . $c;
    }

    return $s;
}
